import { EXIT_CODES } from '../exit-codes.js';
import { which } from '../utils/which.js';
import { exec } from '../utils/exec.js';
import { satisfies } from 'semver';
import { nodeVersion, pnpmVersion } from '../utils/package-json.js';
import { logger } from '../utils/logger.js';

export const checkNodeVersion = () => {
  const pathEnv = process.env.NVM_BIN ? process.env.NVM_BIN : process.env.PATH;
  const nodePath = which('node', pathEnv);

  if (!nodePath) {
    console.error(`❌ Node.js not found. Please install Node.js to continue 🟩. Searched for paths: ${pathEnv}`);
    process.exit(EXIT_CODES.NODE_NOT_FOUND);
  }

  let nodeVersionResult = exec(`"${nodePath}" --version`).trim();
  if (nodeVersionResult.startsWith('v'))
    nodeVersionResult = nodeVersionResult.slice(1);

  if (!satisfies(nodeVersionResult, nodeVersion)) {
    logger.error(`❌ Node.js version not supported: ${nodeVersionResult}. Please install Node.js version satisfying: ${nodeVersion} 🟩`);

    process.exit(EXIT_CODES.NODE_VERSION_NOT_SUPPORTED);
  }

  logger.info(`✅ Node.js version ${nodeVersionResult} is supported 🟩`);
};

export const checkPnpmVersion = () => {
  const pathEnv = process.env.PNPM_HOME ? process.env.PNPM_HOME : process.env.PATH;
  const pnpmPath = which('pnpm', pathEnv);

  if (!pnpmPath) {
    console.error(`❌ pnpm not found. Please install pnpm to continue 📦. Searched for paths: ${pathEnv}`);
    process.exit(EXIT_CODES.PNPM_NOT_FOUND);
  }

  let pnpmVersionResult = exec(`"${pnpmPath}" --version`).trim();
  if (pnpmVersionResult.startsWith('v'))
    pnpmVersionResult = pnpmVersionResult.slice(1);

  if (!satisfies(pnpmVersionResult, pnpmVersion)) {
    logger.error(`❌ Pnpm version not supported: ${pnpmVersionResult}. Please install exact pnpm version: ${pnpmVersion} 📦`);

    process.exit(EXIT_CODES.PNPM_VERSION_NOT_SUPPORTED);
  }

  logger.info(`✅ pnpm version ${pnpmVersionResult} is supported 📦`);
};
