
class ProceduralPublicKey(object):
    """
    A procedural public key to be generated by get_dev_key
    """

    def __init__(self, name):
        self.name = name
        return

class ProceduralPrivateKey(object):
    """
    A procedural private key to be generated by get_dev_key
    """

    def __init__(self, name):
        self.name = name
        return

class ProceduralKeyDatabase(object):
    """
    Every synthetic testnet key is generated by concatenating the name, secret and role.
    This class is the central place these are issued, and keeps track of all of them.
    """
    def __init__(self):
        self.seed2pair = {}
        return

    def get(self, name, role="active"):
        seed = role+"-"+name
        pair = self.seed2pair.get(seed)
        if pair is None:
            pair = [ProceduralPublicKey(seed), ProceduralPrivateKey(seed)]
            self.seed2pair[seed] = pair
        return pair

    def get_pubkey(self, name, role="active"):
        return self.get(name, role)[0]

    def get_privkey(self, name, role="active"):
        return self.get(name, role)[1]

    def get_authority(self, name, role="active"):
        return {
         "weight_threshold":1,
         "account_auths":[],
         "key_auths":[[self.get_pubkey(name, role),1]],
        }

class PubkeySerializer(object):
    """
    We want to defer substitution of public keys to a later
    date.  I.e. we generate an object of a format roughly
    like this:

    {"tx" : {"operations" : [{"type" : "account_create_operation", "value" : {
      "creator" : "alice",
      "new_account_name" : "bob",
      "memo_key" : "bob-memo", ...}}]},
     "signers" : ["alice-active"],
     "pubkeysubs" : ["bob-memo"]
    }

    Then the submitter is responsible for generating the procedural
    keys alice-active and bob-memo, and substituting them in the
    appropriate places.  The challenging problem is that because
    these transactions can contain arbitrary user data, how to ensure
    they are inserted in *only* the appropriate places.

    Some form of escaping is necessary here, but the API provided by
    Python JSON parsing makes it difficult to either extend the JSON
    format to create an entirely new token type, or process all strings
    (allowing for example `{`, `}` to be escaped to `{{`, `}}` and then
    using `str.format()` compatible replacement).

    What we do instead is a two-pass approach:  Serialize as JSON, then
    construct some alphabetical string that does not occur in the
    serialized JSON, and use that as the escape.

    For the first pass, we have no prefix.
    """

    def __init__(self,
        esc="",
        ):
        self.esc = esc
        return

    def __call__(self, obj):
        if isinstance(obj, ProceduralPublicKey):
            return self.esc + "publickey:" + obj.name + self.esc
        if isinstance(obj, ProceduralPrivateKey):
            return self.esc + "privatekey:" + obj.name + self.esc
        return obj
