import type { TPublicKey, TRole } from "@hiveio/wax";
import type { KeyIndex } from "../rpc";
import { getWax } from "../hive/wax";
import { remove0x } from "@metamask/utils";
import { type BIP44Node, getBIP44AddressKeyDeriver } from "@metamask/key-tree";
import type { IBeekeeperUnlockedWallet } from "@hiveio/beekeeper";

const KeyIndexToPathMap = {
  owner: 0,
  active: 1,
  memo: 3,
  posting: 4
} as const satisfies Record<TRole, number>;

const CoinType = 0xBEE;

export const keyIndexToBip44Node = async(keyIndex: KeyIndex): Promise<BIP44Node> => {
  const bip44 = await snap.request({
    method: 'snap_getBip44Entropy',
    params: {
      coinType: CoinType
    }
  });
  // Create a function that takes an index and returns an extended private key for m/44'/3054'/accountIndex'/0/address_index
  const deriveHiveAddress = await getBIP44AddressKeyDeriver(bip44, { account: keyIndex.accountIndex ?? 0, change: 0 });

  const keyIndexType = KeyIndexToPathMap[keyIndex.role];
  if (keyIndexType === undefined)
    throw new Error(`Invalid key index type: ${keyIndex.role}`);

  // Derive the second Hive address, which has a proper index.
  return await deriveHiveAddress(keyIndexType);
};

export const getPublicKeyWifFromKeyIndex = async (keyIndex: KeyIndex): Promise<TPublicKey> => {
  const wax = await getWax();

  const bip44Node = await keyIndexToBip44Node(keyIndex);

  const publicKey = wax.convertRawPublicKeyToWif(remove0x(bip44Node.compressedPublicKey));

  return publicKey;
};

export const getPrivateKeyWifFromKeyIndex = async (keyIndex: KeyIndex): Promise<string> => {
  const wax = await getWax();

  const bip44Node = await keyIndexToBip44Node(keyIndex);

  if (!bip44Node.privateKey)
    throw new Error('No private key found');

  const wif = wax.convertRawPrivateKeyToWif(remove0x(bip44Node.privateKey));

  return wif;
};

export const importPrivateKeyToWallet = async (wallet: IBeekeeperUnlockedWallet, keyIndex: KeyIndex): Promise<TPublicKey> => {
  const privateKeyWif = await getPrivateKeyWifFromKeyIndex(keyIndex);

  const publicKey = await wallet.importKey(privateKeyWif);

  return publicKey;
};
