"""condenser_api trending tag fetching methods"""

from aiocache import cached
from hive.server.common.helpers import (return_error_info, valid_tag, valid_limit)

@return_error_info
@cached(ttl=7200, timeout=1200)
async def get_top_trending_tags_summary(context):
    """Get top 50 trending tags among pending posts."""
    # Same results, more overhead:
    #return [tag['name'] for tag in await get_trending_tags('', 50)]
    sql = """
        SELECT (SELECT category FROM hive_category_data WHERE id = category_id) as category
          FROM hive_posts
         WHERE is_paidout = '0'
      GROUP BY category
      ORDER BY SUM(payout) DESC
         LIMIT 50
    """
    return await context['db'].query_col(sql)

@return_error_info
@cached(ttl=3600, timeout=1200)
async def get_trending_tags(context, start_tag: str = '', limit: int = 250):
    """Get top 250 trending tags among pending posts, with stats."""

    limit = valid_limit(limit, ubound=250)
    start_tag = valid_tag(start_tag or '', allow_empty=True)

    if start_tag:
        seek = """
          HAVING SUM(payout) <= (
            SELECT SUM(payout)
              FROM hive_posts
             WHERE is_paidout = '0'
               AND category_id = (SELECT id FROM hive_category_data WHERE category = :start_tag))
        """
    else:
        seek = ''

    sql = """
      SELECT (SELECT category FROM hive_category_data WHERE id = category_id) as category,
             COUNT(*) AS total_posts,
             SUM(CASE WHEN depth = 0 THEN 1 ELSE 0 END) AS top_posts,
             SUM(payout) AS total_payouts
        FROM hive_posts
       WHERE is_paidout = '0'
    GROUP BY category %s
    ORDER BY SUM(payout) DESC
       LIMIT :limit
    """ % seek

    out = []
    for row in await context['db'].query_all(sql, limit=limit, start_tag=start_tag):
        out.append({
            'name': row['category'],
            'comments': row['total_posts'] - row['top_posts'],
            'top_posts': row['top_posts'],
            'total_payouts': "%.3f HBD" % row['total_payouts']})

    return out
