"""Conf handles reading run-time config and app-level settings."""

import logging
import configargparse

class Conf():
    """ Manages sync/server configuration via args, ENVs, and hive.conf. """
    _args = None

    @classmethod
    def init_argparse(cls, ignore_unknown=False):
        """Read hive config (CLI arg > ENV var > config)"""
        assert not cls._args, "config already read"

        #pylint: disable=line-too-long
        parser = configargparse.get_arg_parser(
            default_config_files=['./hive.conf'])
        add = parser.add

        # runmodes: sync, server, status
        add('mode', nargs='*', default=['sync'])

        # common
        add('--database-url', env_var='DATABASE_URL', required=False, help='database connection url', default='')
        add('--steemd-url', env_var='STEEMD_URL', required=False, help='steemd/jussi endpoint', default='https://api.steemit.com')
        add('--log-level', env_var='LOG_LEVEL', default='INFO')
        add('--dump-config', type=bool, env_var='DUMP_CONFIG', default=False)

        # specific to indexer
        add('--max-workers', type=int, env_var='MAX_WORKERS', help='max workers for batch requests', default=4)
        add('--max-batch', type=int, env_var='MAX_BATCH', help='max chunk size for batch requests', default=50)
        add('--trail-blocks', type=int, env_var='TRAIL_BLOCKS', help='number of blocks to trail head by', default=2)
        add('--disable-sync', type=bool, env_var='DISABLE_SYNC', help='(debug) skip sync and sweep; jump to block streaming', default=False)
        add('--sync-to-s3', type=bool, env_var='SYNC_TO_S3', help='alternative healthcheck for background sync service', default=False)

        # specific to API server
        add('--http-server-port', type=int, env_var='HTTP_SERVER_PORT', default=8080)

        if ignore_unknown:
            # needed for e.g. tests - other args may be present
            args, _ = parser.parse_known_args()
        else:
            args = parser.parse_args()

        cls._args = vars(args)

        root = logging.getLogger()
        root.setLevel(cls.log_level())
        root.info("loaded configuration:\n%s", parser.format_values())

    @classmethod
    def init_config(cls, config):
        """Initialize config directly (do not parse ENV/cli/conf)."""
        cls._args = config

    @classmethod
    def args(cls):
        """Get the raw Namespace object as generated by configargparse"""
        return cls._args

    @classmethod
    def get(cls, param):
        """Reads a single property, e.g. `database_url`."""
        assert cls._args, "run init_argparse()"
        return cls._args[param]

    @classmethod
    def run_mode(cls):
        """Get the (enforced-valid) run mode as a normalized string."""
        mode = '/'.join(cls.get('mode'))
        valid = ['server', 'sync', 'status']
        assert mode in valid, "invalid run mode %s" % mode
        return mode

    @classmethod
    def log_level(cls):
        """Get `logger`s internal int level from config string."""
        str_log_level = cls.get('log_level')
        log_level = getattr(logging, str_log_level.upper(), None)
        if not isinstance(log_level, int):
            raise ValueError('Invalid log level: %s' % str_log_level)
        return log_level
