"""Conf handles reading run-time config and app-level settings."""

import logging
import configargparse

class Conf():
    _args = None

    @classmethod
    def init_argparse(cls):
        """Read hive config (CLI arg > ENV var > config)"""
        assert not cls._args, "config already read"

        #pylint: disable=invalid-name,line-too-long
        p = configargparse.get_arg_parser(default_config_files=['./hive.conf'])

        # runmodes: sync, server, status
        p.add('mode', nargs='*', default=['sync'])

        # common
        p.add('--database-url', env_var='DATABASE_URL', required=True, help='database connection url', default='postgresql://user:pass@localhost:5432/hive')
        p.add('--steemd-url', env_var='STEEMD_URL', required=True, help='steemd/jussi endpoint', default='https://api.steemit.com')
        p.add('--log-level', env_var='LOG_LEVEL', default='INFO')

        # specific to indexer
        p.add('--max-workers', type=int, env_var='MAX_WORKERS', help='max workers for batch requests (untested)', default=1)
        p.add('--max-batch', type=int, env_var='MAX_BATCH', help='max size for batch block/content requests', default=100)
        p.add('--trail-blocks', type=int, env_var='TRAIL_BLOCKS', help='number of blocks to trail head by', default=2)
        p.add('--disable-sync', type=bool, env_var='DISABLE_SYNC', help='(debug) skip sync and sweep; jump to block streaming', default=False)

        # specific to API server
        p.add('--http-server-port', type=int, env_var='HTTP_SERVER_PORT', default=8080)

        cls._args = p.parse_args()

        if cls.get('log_level') == 'DEBUG':
            print(cls._args)
            print(p.format_values())

    @classmethod
    def args(cls):
        """Get the raw Namespace object as generated by configargparse"""
        return cls._args

    @classmethod
    def get(cls, param):
        """Reads a single property, e.g. `database_url`."""
        assert cls._args, "run init_argparse()"
        return getattr(cls._args, param)

    @classmethod
    def run_mode(cls):
        """Get the (enforced-valid) run mode as a normalized string."""
        mode = '/'.join(cls.get('mode'))
        valid = ['server', 'sync', 'status']
        assert mode in valid, "invalid run mode %s" % mode
        return mode

    @classmethod
    def log_level(cls):
        """Get `logger`s internal int level from config string."""
        str_log_level = cls.get('log_level')
        log_level = getattr(logging, str_log_level.upper(), None)
        if not isinstance(log_level, int):
            raise ValueError('Invalid log level: %s' % str_log_level)
        return log_level
