# coding=utf-8
import sys
import os

from setuptools import find_packages
from setuptools import setup
from setuptools.command.install import install
from setuptools.command.build_py import build_py as build_py

assert sys.version_info[0] == 3 and sys.version_info[1] >= 6, "hive requires Python 3.6 or newer"

VERSION = '0.0.1'

class GitRevisionProvider(object):
    """ Static class to provide version and git revision information"""

    @staticmethod
    def provide_git_revision():
        """ Evaluate version and git revision and save it to a version file
            Evaluation is based on VERSION variable and git describe if
            .git directory is present in tree.
            In case when .git is not available version and git_revision is taken
            from get_distribution call
        """
        if os.path.exists(".git"):
            from subprocess import check_output
            command = 'git describe --tags --long --dirty'
            version = check_output(command.split()).decode('utf-8').strip()
            parts = version.split('-')
            _, _, sha = parts[:3]
            GitRevisionProvider._save_version_file(VERSION, sha.lstrip('g'))
        else:
            from pkg_resources import get_distribution
            try:
                version, git_revision = get_distribution("hivemind").version.split("+")
                GitRevisionProvider._save_version_file(version, git_revision)
            except:
                GitRevisionProvider._save_version_file(VERSION, "")

    @staticmethod
    def _save_version_file(hivemind_version, git_revision):
        """ Helper method to save version.py with current version and git_revision """
        with open("hive/version.py", 'w') as version_file:
            version_file.write("# generated by setup.py\n")
            version_file.write("# contents will be overwritten\n")
            version_file.write("VERSION = '{}'\n".format(hivemind_version))
            version_file.write("GIT_REVISION = '{}'".format(git_revision))

class BuildWrapper(build_py):
    def run(self):
        GitRevisionProvider.provide_git_revision()
        build_py.run(self)

class InstallWrapper(install):
    def run(self):
        GitRevisionProvider.provide_git_revision()
        install.do_egg_install(self)

# yapf: disable
setup(
    name='hivemind',
    version_format=VERSION + "+{gitsha}",
    description='Developer-friendly microservice powering social networks on the Steem blockchain.',
    long_description=open('README.md').read(),
    packages=find_packages(exclude=['scripts']),
    setup_requires=[
        'pytest-runner',
        'setuptools-git-version'
    ],
    install_requires=[
        'importlib_metadata',
        'aiopg',
        'jsonrpcserver',
        'simplejson',
        'aiohttp',
        'certifi',
        'sqlalchemy',
        'funcy',
        'toolz',
        'maya',
        'ujson',
        'urllib3',
        'psycopg2-binary',
        'aiocache',
        'configargparse',
        'pdoc',
        'diff-match-patch',
    ],
    entry_points={
        'console_scripts': [
            'hive=hive.cli:run',
        ]
    },
    cmdclass={
        'install' : InstallWrapper,
        'build_py' : BuildWrapper
    }
)
