from __future__ import annotations

import json
from pathlib import Path
from typing import TYPE_CHECKING, Any, ClassVar, TypeVar

import msgspec

from beekeepy._interface.url import HttpUrl, Url
from schemas._preconfigured_base_model import PreconfiguredBaseModel

if TYPE_CHECKING:
    from typing_extensions import Self


class Config(PreconfiguredBaseModel):
    DEFAULT_FILE_NAME: ClassVar[str] = "config.ini"

    def save(self, destination: Path) -> None:
        destination = destination / Config.DEFAULT_FILE_NAME if destination.is_dir() else destination
        with destination.open("wt", encoding="utf-8") as out_file:
            out_file.write("# config automatically generated by clive\n")
            for member_name, member_value in msgspec.structs.asdict(self).items():
                if member_value is not None:
                    out_file.write(
                        f"{self._convert_member_name_to_config_name(member_name)}={self._convert_member_value_to_config_value(member_value)}\n"
                    )

    @classmethod
    def load(cls, source: Path) -> Self:
        source = source / Config.DEFAULT_FILE_NAME if source.is_dir() else source
        assert source.exists(), "Given file does not exists."
        values_to_write = {}
        with source.open("rt", encoding="utf-8") as in_file:
            for line in in_file:
                if (line := line.strip("\n")) and not line.startswith("#"):
                    config_name, config_value = line.split("=")
                    member_name = cls._convert_config_name_to_member_name(config_name)
                    member_value = cls._convert_config_value_to_member_value(config_value)
                    values_to_write[member_name] = member_value

        json_values_to_write = json.dumps(values_to_write)
        decoder = get_config_decoder(cls)
        return decoder.decode(json_values_to_write)

    @classmethod
    def _convert_member_name_to_config_name(cls, member_name: str) -> str:
        return member_name.replace("_", "-")

    @classmethod
    def _convert_config_name_to_member_name(cls, config_name: str) -> str:
        return config_name.strip().replace("-", "_")

    @classmethod
    def _convert_member_value_to_config_value(cls, member_value: Any) -> str:
        if isinstance(member_value, list):
            return " ".join(member_value)

        if isinstance(member_value, bool):
            return "yes" if member_value else "no"

        if isinstance(member_value, Url):
            return member_value.as_string(with_protocol=False)

        if isinstance(member_value, Path):
            return member_value.as_posix()
        return str(member_value)

    @classmethod
    def _convert_config_value_to_member_value(cls, config_value: str) -> Any | None:
        config_value = config_value.strip()
        if config_value.isdigit():
            return int(config_value)
        return config_value


def dec_hook_config(type_: type, obj: Any) -> Any:
    if type_ is Path:
        return Path(obj.replace('"', ""))
    if type_ is HttpUrl:
        return HttpUrl(obj)
    raise NotImplementedError(f"Objects of type {type_} are not supported")


T = TypeVar("T")


def get_config_decoder(type_: type[T]) -> msgspec.json.Decoder[T]:
    return msgspec.json.Decoder(type_, dec_hook=dec_hook_config)
