from __future__ import annotations

from pathlib import Path
from types import UnionType
from typing import TYPE_CHECKING, Any, ClassVar, get_args

from loguru import logger
from pydantic import BaseModel

from beekeepy._communication import Url
from beekeepy.exceptions import InvalidOptionError

if TYPE_CHECKING:
    from typing_extensions import Self


class Config(BaseModel):
    DEFAULT_FILE_NAME: ClassVar[str] = "config.ini"

    class Config:
        arbitrary_types_allowed = True

    def save(self, destination: Path) -> None:
        destination = destination / Config.DEFAULT_FILE_NAME if destination.is_dir() else destination
        with destination.open("wt", encoding="utf-8") as out_file:
            out_file.write("# config automatically generated by helpy\n")
            for line in self.write_to_lines():
                out_file.write(line + "\n")

    def write_to_lines(self) -> list[str]:
        result = []
        for member_name, member_value in self.__dict__.items():
            if member_value is not None:
                if isinstance(member_value, list) and len(member_value) == 0:
                    continue

                entry_name = self._convert_member_name_to_config_name(member_name)
                entry_value = self._convert_member_value_to_config_value(member_name, member_value)
                for value in [entry_value] if not isinstance(entry_value, list) else entry_value:
                    result.append(f"{entry_name} = {value}")  # noqa: PERF401  # would be unreadable
        return result

    @classmethod
    def load(cls, source: Path) -> Self:
        source = source / Config.DEFAULT_FILE_NAME if source.is_dir() else source
        assert source.exists(), "Given file does not exists."
        return cls.load_from_lines(source.read_text().strip().splitlines())

    @classmethod
    def load_from_lines(cls, lines: list[str]) -> Self:
        fields = cls.__fields__
        values_to_write: dict[str, Any] = {}
        for line in lines:
            if (line := line.strip("\n")) and not line.startswith("#"):
                config_name, config_value = line.split("=")
                member_name = cls._convert_config_name_to_member_name(config_name)
                member_type = fields[member_name].annotation
                if isinstance(member_type, UnionType) and (type(None) in get_args(member_type)):
                    member_type = get_args(member_type)[0]
                values_to_write[member_name] = cls._convert_config_value_to_member_value(
                    config_value, expected=member_type, current_value=values_to_write.get(member_name)
                )
        return cls(**values_to_write)

    @classmethod
    def _convert_member_name_to_config_name(cls, member_name: str) -> str:
        return member_name.replace("_", "-")

    @classmethod
    def _convert_config_name_to_member_name(cls, config_name: str) -> str:
        return config_name.strip().replace("-", "_")

    @classmethod
    def _convert_member_value_to_config_value(cls, member_name: str, member_value: Any) -> str | list[str]:  # noqa: ARG003
        if isinstance(member_value, list):
            return member_value

        if isinstance(member_value, bool):
            return "yes" if member_value else "no"

        if isinstance(member_value, Url):
            return member_value.as_string(with_protocol=False)

        if isinstance(member_value, Path):
            return member_value.as_posix()

        return str(member_value)

    @classmethod
    def _convert_config_value_to_member_value(  # noqa: PLR0911, C901
        cls, config_value: str, *, expected: type[Any], current_value: Any | None
    ) -> Any | None:
        config_value = config_value.strip()
        if config_value is None:
            return None

        if expected == Path:
            return Path(config_value.replace('"', ""))

        if issubclass(expected, list) or "list" in str(expected):
            list_arg_t = get_args(expected)[0]
            if len(get_args(list_arg_t)):  # in case of unions
                list_arg_t = get_args(list_arg_t)[0]
            logger.info(f"{list_arg_t=}")
            values = [
                cls._convert_config_value_to_member_value(value, expected=list_arg_t, current_value=None)
                for value in config_value.split()
            ]
            if current_value is not None:
                if isinstance(current_value, list):
                    current_value.extend(values)
                    return current_value
                return [*values, current_value]
            return values

        if expected == Url:
            return Url(config_value)

        if expected is type(bool):
            cv_lower = config_value.lower()
            if cv_lower == "yes":
                return True

            if cv_lower == "no":
                return False

            raise InvalidOptionError(f"Expected `yes` or `no`, got: `{config_value}`")

        if "str" in str(expected):
            return config_value.strip('"')

        if isinstance(expected, type) and issubclass(expected, int | str) and hasattr(expected, "validate"):
            return expected.validate(config_value)

        return expected(config_value) if expected is not None else None

    def get_differences_between(self, other: Self) -> dict[str, tuple[Any, Any]]:
        differences = {}
        for member_name in self.__dict__:
            self_value = getattr(self, member_name)
            other_value = getattr(other, member_name)
            if self_value != other_value:
                differences[member_name] = (self_value, other_value)
        return differences
