#pragma once

#include "pxx_new_types.hpp"

#include "psql_utils/pg_cxx.hpp"

#include "postgres.h"
#include "fmgr.h"
#include "utils/builtins.h"
#include "utils/timestamp.h"

#include <string>
#include <cstddef>
#include <cstdint>
#include <string>




#define spixx_elog(elevel, ...)  \
	ereport(elevel, errmsg_internal(__VA_ARGS__))


typedef uintptr_t Datum;
struct SPITupleTable;

extern "C"
{
  struct TupleDescData;
  typedef TupleDescData* TupleDesc;
  struct HeapTupleData;
  typedef HeapTupleData* HeapTuple;
  int SPI_connect();
  int SPI_finish();
} // extern "C"



namespace spixx
{

struct field
{
  Datum datum;
  bool isNull;

  bool is_null() const noexcept
  {
    return isNull;
  }

  const char * c_str() const
  {
    if (isNull)
    {
      return "";
    }

    return text_to_cstring(DatumGetTextP(datum));

  }

  template<typename T> T as() const;

};



template<>
inline std::string_view field::as<std::string_view>() const
{

  text* text_ptr = DatumGetTextP(datum);

  char* cstr = VARDATA(text_ptr);
  int length = VARSIZE(text_ptr) - VARHDRSZ;

  std::string_view my_string_view(cstr, length);
  return my_string_view;
}


template<>
inline std::basic_string<std::byte> field::as<std::basic_string<std::byte>>() const
{
  bytea *bytea_data = DatumGetByteaP(datum);
  char *data = VARDATA(bytea_data);
  size_t size = VARSIZE(bytea_data) - VARHDRSZ;

  std::basic_string<std::byte> byte_string(reinterpret_cast<std::byte *>(data), size);

  return byte_string;
}



class row
{
protected:
  HeapTuple tuple;

private:
  TupleDesc tupdesc;

public:
  row(HeapTuple t, TupleDesc td) : tuple(t), tupdesc(td) {}

  field operator[](const std::string &key) const
  {
    int col = SPI_fnumber(tupdesc, key.c_str());
    if (col <= 0)
    {
      spixx_elog(ERROR, "Column not found");
    }
    bool isN;
    Datum datum = SPI_getbinval(tuple, tupdesc, col, &isN);
    return field{datum, isN};
  }

  friend void display_type_info(const row& r, const std::string &key);

};

class const_result_iterator
{
private:
  SPITupleTable* tuptable;
  int index;

public:
  explicit const_result_iterator(SPITupleTable* tt, int idx);
  const_result_iterator();

  const_result_iterator& operator++();

  [[nodiscard]] bool operator!=(const_result_iterator const& i) const;
  [[nodiscard]] bool operator==(const_result_iterator const& i) const;
  row operator*() const;
};

class result
{
private:
  SPITupleTable* tuptable;
  uint64_t proc;

public:
  result();
  result(SPITupleTable* t, TupleDesc td, uint64_t p);

  using const_iterator = const_result_iterator;
  [[nodiscard]] const_iterator end() const noexcept;
  [[nodiscard]] bool empty() const noexcept;
  [[nodiscard]] const_iterator begin() const noexcept;
  row operator[](size_t i) const noexcept;
  
  friend void display_column_names_and_types(const result& recordset, const std::string &label);

};


result execute_query(const std::string& query);

template<>
inline pxx_new_types::timestamp_wo_tz_type field::as<pxx_new_types::timestamp_wo_tz_type>() const
{

  Timestamp timestamp = DatumGetTimestamp(datum);

  char* timestamp_string = DatumGetCString(DirectFunctionCall1(timestamp_out, TimestampGetDatum(timestamp)));

  auto res = pxx_new_types::timestamp_wo_tz_type{ std::string(timestamp_string) };

  pfree(timestamp_string);
  return res;
}

template<>
inline pxx_new_types::jsonb_string field::as<pxx_new_types::jsonb_string>() const
{
  std::string s;

  if(!isNull)
  {
    Jsonb* jb = DatumGetJsonbP(datum);

    char* j_string = JsonbToCString(NULL, &jb->root, VARSIZE(jb));

    s = (j_string);
    pfree(j_string);
  }

  auto res = pxx_new_types::jsonb_string{ s };

  return res;
}

} // namespace spixx

