import React from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import tt from 'counterpart';
import * as appActions from 'app/redux/AppReducer';
import CloseButton from 'app/components/elements/CloseButton';
import Icon from 'app/components/elements/Icon';
import { Link } from 'react-router';

const SidePanel = ({
 alignment, visible, hideSidePanel, username, walletUrl, toggleNightmode
}) => {
    if (process.env.BROWSER) {
        if (visible) {
            document.addEventListener('click', hideSidePanel);
        } else {
            document.removeEventListener('click', hideSidePanel);
        }
    }

    const loggedIn = username === undefined ? 'show-for-small-only' : 'SidePanel__hide-signup';

    const makeLink = (i, ix, arr) => {
        // A link is internal if it begins with a slash
        const isExternal = !i.link.match(/^\//) || i.isExternal;
        const cn = ix === arr.length - 1 ? 'last' : null;
        if (isExternal) {
            return (
                <li key={ix} className={cn}>
                    <a href={i.link} target="_blank" rel="noopener noreferrer">
                        {i.label}
                        &nbsp;
                        <Icon name="extlink" />
                    </a>
                </li>
            );
        }
        if (i.onClick) {
            return (
                <li key={ix} className={cn}>
                    <a role="link" tabIndex={0} onClick={i.onClick}>{i.label}</a>
                </li>
            );
        }
        return (
            <li key={ix} className={cn}>
                <Link to={i.link}>{i.label}</Link>
            </li>
        );
    };

    const sidePanelLinks = {
        internal: [
            {
                label: tt('navigation.welcome'),
                link: `/welcome`,
            },
            {
                label: tt('navigation.faq'),
                link: `/faq.html`,
            },
            {
                label: tt('navigation.block_explorer'),
                link: `https://hiveblocks.com`,
            },
            /*
            {
                label: tt('navigation.explore'),
                link: `/communities`,
            },
            */
            {
                label: tt('g.toggle_nightmode'),
                link: '/',
                onClick: toggleNightmode,
            },
        ],

        wallet: [
            {
                label: tt('navigation.stolen_account_recovery'),
                link: `${walletUrl}/recover_account_step_1`,
            },
            {
                label: tt('navigation.change_account_password'),
                link: `${walletUrl}/change_password`,
            },
            {
                label: tt('navigation.vote_for_witnesses'),
                link: `${walletUrl}/~witnesses`,
            },
            {
                label: tt('navigation.hive_proposals'),
                link: `${walletUrl}/proposals`,
            },
        ],

        external: [
            {
                label: tt('navigation.chat'),
                link: 'https://openhive.chat/home',
            },
        ],

        organizational: [
            {
                label: tt('navigation.api_docs'),
                link: 'https://developers.hive.io/',
            },
            /*
            {
                label: tt('navigation.bluepaper'),
                link: 'https://hive.io/hive-bluepaper.pdf',
            },
            {
                label: tt('navigation.smt_whitepaper'),
                link: 'https://hive.io/hive-smt-whitepaper.pdf',
            },
            */
            {
                label: tt('navigation.whitepaper'),
                link: 'https://hive.io/whitepaper.pdf',
            },
        ],

        legal: [
            {
                label: tt('navigation.privacy_policy'),
                link: '/privacy.html',
            },
            {
                label: tt('navigation.terms_of_service'),
                link: '/tos.html',
            },
        ],

        extras: [
            {
                label: tt('g.sign_in'),
                link: '/login.html',
            },
            {
                label: tt('g.sign_up'),
                link: 'https://signup.hive.io',
            },
        ],
    };

    return (
        <div className="SidePanel">
            <div className={(visible ? 'visible ' : '') + alignment}>
                <CloseButton onClick={hideSidePanel} />
                <ul className={`vertical menu ${loggedIn}`}>{sidePanelLinks.extras.map(makeLink)}</ul>
                <ul className="vertical menu">{sidePanelLinks.internal.map(makeLink)}</ul>
                <ul className="vertical menu">{sidePanelLinks.wallet.map(makeLink)}</ul>
                <ul className="vertical menu">{sidePanelLinks.external.map(makeLink)}</ul>
                <ul className="vertical menu">{sidePanelLinks.organizational.map(makeLink)}</ul>
                <ul className="vertical menu">{sidePanelLinks.legal.map(makeLink)}</ul>
            </div>
        </div>
    );
};

SidePanel.propTypes = {
    alignment: PropTypes.oneOf(['left', 'right']).isRequired,
    visible: PropTypes.bool.isRequired,
    hideSidePanel: PropTypes.func.isRequired,
    username: PropTypes.string,
    toggleNightmode: PropTypes.func.isRequired,
};

SidePanel.defaultProps = {
    username: undefined,
};

export default connect(
    (state, ownProps) => {
        const walletUrl = state.app.get('walletUrl');
        return {
            walletUrl,
            ...ownProps,
        };
    },
    (dispatch) => ({
        toggleNightmode: (e) => {
            if (e) e.preventDefault();
            dispatch(appActions.toggleNightmode());
        },
    })
)(SidePanel);
