from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING

import beekeepy.exceptions as bke

from clive.__private.core.commands.abc.command_secured import CommandPasswordSecured
from clive.__private.core.commands.abc.command_with_result import CommandWithResult
from clive.__private.core.commands.migrate_profile import MigrateProfile
from clive.__private.core.commands.recover_wallets import RecoverWallets, RecoverWalletsStatus
from clive.__private.core.commands.set_timeout import SetTimeout
from clive.__private.core.encryption import EncryptionService
from clive.__private.core.wallet_container import WalletContainer

if TYPE_CHECKING:
    from datetime import timedelta

    from beekeepy import AsyncSession, AsyncUnlockedWallet

    from clive.__private.core.app_state import AppState
    from clive.__private.core.types import MigrationStatus


@dataclass
class UnlockWalletStatus:
    recovery_status: RecoverWalletsStatus
    migration_status: MigrationStatus


@dataclass(kw_only=True)
class Unlock(CommandPasswordSecured, CommandWithResult[UnlockWalletStatus]):
    """
    Unlock the profile-related wallets (user keys and encryption key) managed by the beekeeper.

    Attributes:
        profile_name: The name of the profile to unlock.
        session: The session in which wallets should be unlocked.
        time: Will set a timeout for the profile. Doesn't matter if `permanent` is True.
            Required when `permanent` is False.
        permanent: If True, the timeout will be permanent; otherwise, `time` should also be given.
        app_state: The application state to unlock, if available.
    """

    profile_name: str
    session: AsyncSession
    time: timedelta | None = None
    permanent: bool = True
    app_state: AppState | None = None

    async def _execute(self) -> None:
        await SetTimeout(session=self.session, time=self.time, permanent=self.permanent).execute()

        encryption_wallet = await self._unlock_wallet(EncryptionService.get_encryption_wallet_name(self.profile_name))
        user_wallet = await self._unlock_wallet(self.profile_name)
        is_encryption_wallet_missing = encryption_wallet is None
        is_user_wallet_missing = user_wallet is None

        recover_wallets_result = await RecoverWallets(
            password=self.password,
            profile_name=self.profile_name,
            session=self.session,
            should_recover_encryption_wallet=is_encryption_wallet_missing,
            should_recover_user_wallet=is_user_wallet_missing,
        ).execute_with_result()

        recovery_status = recover_wallets_result.status

        if recovery_status == "encryption_wallet_recovered":
            encryption_wallet = recover_wallets_result.recovered_wallet
        elif recovery_status == "user_wallet_recovered":
            user_wallet = recover_wallets_result.recovered_wallet

        assert user_wallet is not None, "User wallet should be created at this point"
        assert encryption_wallet is not None, "Encryption wallet should be created at this point"

        if self.app_state is not None:
            await self.app_state.unlock(WalletContainer(user_wallet, encryption_wallet))

        migration_result = await MigrateProfile(
            profile_name=self.profile_name,
            unlocked_wallet=user_wallet,
            unlocked_encryption_wallet=encryption_wallet,
        ).execute_with_result()

        self._result = UnlockWalletStatus(recovery_status=recovery_status, migration_status=migration_result)

    async def _unlock_wallet(self, name: str) -> AsyncUnlockedWallet | None:
        try:
            wallet = await self.session.open_wallet(name=name)
        except bke.NoWalletWithSuchNameError:
            return None
        return await wallet.unlock(password=self.password)
