from __future__ import annotations

from typing import TYPE_CHECKING, cast

import typer

from clive.__private.cli.clive_typer import CliveTyper
from clive.__private.cli.common import options

if TYPE_CHECKING:
    from clive.__private.models.asset import Asset

delegations = CliveTyper(name="delegations", help="Set or remove vesting delegaton.")

_delegatee_account_name = typer.Option(
    ...,
    "--delegatee",
    help='The account to use as "delegatee" argument.',
)


@delegations.command(name="set")
async def process_delegations_set(  # noqa: PLR0913
    account_name: str = options.account_name,
    delegatee: str = _delegatee_account_name,
    amount: str = options.voting_amount,
    sign_with: str | None = options.sign_with,
    autosign: bool | None = options.autosign,  # noqa: FBT001
    broadcast: bool | None = options.broadcast,  # noqa: FBT001
    save_file: str | None = options.save_file,
    force: bool = options.force,  # noqa: FBT001
) -> None:
    """Add or modify vesting shares delegation for pair of accounts "account-name" and "delegatee"."""
    from clive.__private.cli.commands.process.process_delegations import ProcessDelegations  # noqa: PLC0415

    amount_ = cast("Asset.VotingT", amount)
    operation = ProcessDelegations(
        delegator=account_name,
        delegatee=delegatee,
        amount=amount_,
        sign_with=sign_with,
        broadcast=broadcast,
        save_file=save_file,
        force=force,
        autosign=autosign,
    )
    await operation.run()


@delegations.command(name="remove")
async def process_delegations_remove(  # noqa: PLR0913
    account_name: str = options.account_name,
    delegatee: str = _delegatee_account_name,
    sign_with: str | None = options.sign_with,
    autosign: bool | None = options.autosign,  # noqa: FBT001
    broadcast: bool | None = options.broadcast,  # noqa: FBT001
    save_file: str | None = options.save_file,
) -> None:
    """Clear vesting shares delegation (by setting it to zero) for pair of accounts "account-name" and "delegatee"."""
    from clive.__private.cli.commands.process.process_delegations import ProcessDelegationsRemove  # noqa: PLC0415

    operation = ProcessDelegationsRemove(
        delegator=account_name,
        delegatee=delegatee,
        sign_with=sign_with,
        broadcast=broadcast,
        save_file=save_file,
        autosign=autosign,
    )
    await operation.run()
