from __future__ import annotations

import typer

from clive.__private.cli.clive_typer import CliveTyper
from clive.__private.cli.common.parameters.argument_related_options import _make_argument_related_option
from clive.__private.cli.common.parameters.argument_related_options import (
    account_name as account_name_argument_related_option,
)
from clive.__private.cli.common.parameters.ensure_single_value import EnsureSingleValue
from clive.__private.cli.common.parameters.styling import stylized_help
from clive.__private.core.types import AuthorityLevel

generate = CliveTyper(name="generate", help="Commands for generating things (e.g. keys).")


_account_name_argument = typer.Argument(
    None,
    help=stylized_help(
        "Account for which key is derived, this is not working account of profile.", required_as_arg_or_option=True
    ),
)

_role_argument = typer.Argument(
    None,
    help=stylized_help("Role for which key is derived.", required_as_arg_or_option=True),
)


@generate.command(name="key-from-seed")
async def generate_key_from_seed(  # noqa: PLR0913
    account_name: str | None = _account_name_argument,
    account_name_option: str | None = account_name_argument_related_option,
    role: AuthorityLevel | None = _role_argument,
    role_option: AuthorityLevel | None = _make_argument_related_option("--role"),
    only_private_key: bool = typer.Option(  # noqa: FBT001
        default=False, help="Whether to display only the private key instead of key pair."
    ),
    only_public_key: bool = typer.Option(  # noqa: FBT001
        default=False, help="Whether to display only the public key instead of key pair."
    ),
) -> None:
    """
    Derive private key by hashing account-name, role and seed.

    Seed (like a secret phrase) should be given at stdin (possibly in a non interactive mode i.e. piped).
    This can be for example phrase generated by `clive generate secret-phrase`.
    """
    from clive.__private.cli.commands.generate.generate_key_from_seed import GenerateKeyFromSeed  # noqa: PLC0415

    await GenerateKeyFromSeed(
        account_name=EnsureSingleValue("account-name").of(account_name, account_name_option),
        role=EnsureSingleValue[AuthorityLevel]("role").of(role, role_option),
        only_private_key=only_private_key,
        only_public_key=only_public_key,
    ).run()


@generate.command(name="public-key")
async def generate_public_key() -> None:
    """Display the public key that corresponds to the private key provided at stdin."""
    from clive.__private.cli.commands.generate.generate_public_key import GeneratePublicKey  # noqa: PLC0415

    await GeneratePublicKey().run()


@generate.command(name="random-key")
async def generate_random_key(
    key_pairs: int = typer.Argument(default=1, help="Amount of key pairs to generate."),
    key_pairs_option: int | None = _make_argument_related_option("--key-pairs"),
) -> None:
    """Generate random key pairs (a private key and its corresponding public key)."""
    from clive.__private.cli.commands.generate.generate_random_key import GenerateRandomKey  # noqa: PLC0415

    await GenerateRandomKey(key_pairs=EnsureSingleValue[int]("key-pairs").of(key_pairs, key_pairs_option)).run()


@generate.command(name="secret-phrase")
async def generate_secret_phrase() -> None:
    """
    Suggest 16 words (brain key) that can be used to derive private keys.

    The secret phrase can be stored offline, and it's easier to store it than all private keys
    in wallet import format (wif). Private key is derived from hashing brain key + account name + role.
    Knowing secret key means knowing all other keys.
    For example use command `clive generate key-from-seed --account-name alice --role active`
    and enter brain key as stdin to get active private key enerated from this secret key.
    """
    from clive.__private.cli.commands.generate.generate_secret_phrase import GenerateSecretPhrase  # noqa: PLC0415

    await GenerateSecretPhrase().run()
