from __future__ import annotations

import typer

from clive.__private.cli.clive_typer import CliveTyper
from clive.__private.cli.common import argument_related_options, modified_param
from clive.__private.cli.common.parameters.ensure_single_value import EnsureSingleAccountNameValue
from clive.__private.cli.common.parameters.styling import stylized_help

known_account = CliveTyper(name="known-account", help="Manage your known account(s).")

_account_name_add_argument = typer.Argument(
    None, help=stylized_help("The name of the known account to add.", required_as_arg_or_option=True)
)


@known_account.command(name="add")
async def add_known_account(
    account_name: str | None = _account_name_add_argument,
    account_name_option: str | None = argument_related_options.account_name,
) -> None:
    """Add an account to the list of known accounts."""
    from clive.__private.cli.commands.configure.known_account import AddKnownAccount  # noqa: PLC0415

    await AddKnownAccount(account_name=EnsureSingleAccountNameValue().of(account_name, account_name_option)).run()


_account_name_remove_argument = modified_param(
    _account_name_add_argument,
    help=stylized_help("The name of the known account to remove.", required_as_arg_or_option=True),
)


@known_account.command(name="remove")
async def remove_known_account(
    account_name: str | None = _account_name_remove_argument,
    account_name_option: str | None = argument_related_options.account_name,
) -> None:
    """Remove an account from the list of known accounts."""
    from clive.__private.cli.commands.configure.known_account import RemoveKnownAccount  # noqa: PLC0415

    await RemoveKnownAccount(account_name=EnsureSingleAccountNameValue().of(account_name, account_name_option)).run()


@known_account.command(name="enable")
async def enable_known_accounts(
    ctx: typer.Context,  # noqa: ARG001
) -> None:
    """
    CLI - If you want to broadcast an operation, you must first add the target account to the list of known accounts.
    TUI - The target account is added to the list of known accounts automatically after adding an operation to the cart.
    """  # noqa: D205
    from clive.__private.cli.commands.configure.known_account import EnableKnownAccounts  # noqa: PLC0415

    await EnableKnownAccounts().run()


@known_account.command(name="disable")
async def disable_know_accounts(
    ctx: typer.Context,  # noqa: ARG001
) -> None:
    """
    CLI - The target account is not checked if it is on the list of known accounts.
    TUI - The target account is not added to the list of known accounts automatically\
 after adding an operation to the cart.
    """  # noqa: D205
    from clive.__private.cli.commands.configure.known_account import DisableKnownAccounts  # noqa: PLC0415

    await DisableKnownAccounts().run()
