from __future__ import annotations

import errno
import inspect
import sys
from abc import ABC, abstractmethod
from dataclasses import Field, dataclass, field, fields
from typing import TYPE_CHECKING, Any, Self

import typer

from clive.__private.cli.exceptions import CLIMutuallyExclusiveOptionsError, CLIPrettyError, CLIRequiresInteractiveError
from clive.__private.core.accounts.exceptions import NoWorkingAccountError
from clive.__private.core.constants.cli import PERFORM_WORKING_ACCOUNT_LOAD

if TYPE_CHECKING:
    from clive.__private.core.profile import Profile


@dataclass(kw_only=True)
class ExternalCLICommand(ABC):
    _skip_validation: bool = field(default=False, init=False)

    @abstractmethod
    async def _run(self) -> None:
        """Actual implementation of the command."""

    @property
    def is_interactive(self) -> bool:
        return sys.stdin.isatty()

    async def run(self) -> None:
        if not self._skip_validation:
            await self.validate()
        await self._configure()
        await self._run()
        await self.post_run()

    async def post_run(self) -> None:
        """Performed after _run, can be overridden in subclass."""
        return

    async def validate(self) -> None:
        """
        Validate the command before running.

        If the command is invalid, raise an CLIPrettyError (or it's derivative) exception.
        """
        self.validate_all_mutually_exclusive_options()

    def validate_all_mutually_exclusive_options(self) -> None:
        """All mutually exclusive options should be validated here."""
        return

    async def _configure(self) -> None:
        """Configure the command before running."""
        return

    @classmethod
    def from_(cls, **kwargs: Any) -> Self:
        """
        Create an instance of a command from the given kwargs.

        Unused kwargs are ignored.

        Args:
            **kwargs: The kwargs to create the instance from.

        Returns:
            An instance of the command class with the provided kwargs.
        """
        sanitized = {k: v for k, v in kwargs.items() if k in inspect.signature(cls).parameters}
        return cls(**sanitized)

    @classmethod
    def is_option_given(cls, value: object) -> bool:
        return value is not None

    def read_interactive(self, prompt: str, *, hide_input: bool = True) -> str:
        """
        Read input in interactive mode.

        Args:
            prompt: Message displayed to the user.
            hide_input: If True, hides the input (e.g., for passwords).

        Raises:
            CLIRequiresInteractiveError: If not in interactive mode.

        Returns:
            The string read from stdin.
        """
        if not self.is_interactive:
            raise CLIRequiresInteractiveError("read input")
        input_value = typer.prompt(prompt, hide_input=hide_input, type=str)
        return str(input_value)

    def read_piped(self) -> str:
        return sys.stdin.readline().rstrip()

    def _validate_mutually_exclusive(self, *, details: str = "", **options: bool) -> None:
        truthy = [name for name, value in options.items() if value]
        if len(truthy) > 1:
            raise CLIMutuallyExclusiveOptionsError(truthy[0], *truthy[1:], details=details)

    def __get_initialized_fields(self) -> list[Field[Any]]:
        return [field_instance for field_instance in fields(self) if field_instance.init]

    def _supply_with_correct_default_for_working_account(self, profile: Profile) -> None:
        """
        We can load default working account value only during runtime.

        Profile name must be known when loading working account as working account is part of profile.
        """

        def get_working_account_name() -> str:
            try:
                return profile.accounts.working.name
            except NoWorkingAccountError as err:
                raise CLIPrettyError(
                    "Working account is not set, can't use working account as default.",
                    errno.EINVAL,
                ) from err

        for field_instance in self.__get_initialized_fields():
            field_name = field_instance.name
            field_value = getattr(self, field_name)
            if field_value == PERFORM_WORKING_ACCOUNT_LOAD:
                setattr(self, field_name, get_working_account_name())
            elif isinstance(field_value, list) and PERFORM_WORKING_ACCOUNT_LOAD in field_value:
                new_attr = [
                    get_working_account_name() if elem == PERFORM_WORKING_ACCOUNT_LOAD else elem for elem in field_value
                ]
                setattr(self, field_name, new_attr)
