from __future__ import annotations

from pathlib import Path
from typing import TYPE_CHECKING

from pydantic import Field

from clive.__private.core.beekeeper.defaults import BeekeeperDefaults
from clive.__private.core.url import Url
from clive.__private.models.base import CliveBaseModel
from clive.__private.settings import safe_settings
from clive.exceptions import CliveError

if TYPE_CHECKING:
    from clive.__private.core.beekeeper.command_line_args import ExportKeysWalletParams

AllowedTypesT = str | list[str] | bool | int | Url | Path


class InvalidOptionError(CliveError):
    pass


def webserver_default() -> Url:
    return Url("http", "127.0.0.1", 0)


def _wallet_dir_default() -> Path:
    return Path(safe_settings.data_path) / "beekeeper"


class BeekeeperConfig(CliveBaseModel):
    wallet_dir: Path = Field(default_factory=_wallet_dir_default)
    unlock_timeout: int = BeekeeperDefaults.DEFAULT_UNLOCK_TIMEOUT
    unlock_interval: int = BeekeeperDefaults.DEFAULT_UNLOCK_INTERVAL
    log_json_rpc: Path | None = BeekeeperDefaults.DEFAULT_LOG_JSON_RPC
    webserver_http_endpoint: Url | None = Field(default_factory=webserver_default)
    webserver_unix_endpoint: Url | None = None
    webserver_ws_endpoint: Url | None = None
    webserver_ws_deflate: int = 0
    webserver_thread_pool_size: int = 1
    notifications_endpoint: Url | None = BeekeeperDefaults.DEFAULT_NOTIFICATIONS_ENDPOINT
    backtrace: str = BeekeeperDefaults.DEFAULT_BACKTRACE
    plugin: list[str] = Field(default_factory=lambda: ["json_rpc", "webserver"])
    export_keys_wallet: ExportKeysWalletParams | None = BeekeeperDefaults.DEFAULT_EXPORT_KEYS_WALLET

    class Config:
        arbitrary_types_allowed = True

    @staticmethod
    def get_wallet_dir() -> Path:
        return _wallet_dir_default()

    def save(self, destination: Path) -> None:
        with destination.open("wt", encoding="utf-8") as out_file:
            out_file.write("# config automatically generated by clive\n")
            for member_name, member_value in self.__dict__.items():
                if member_value is not None:
                    out_file.write(
                        f"{self.__convert_member_name_to_config_name(member_name)}={self.__convert_member_value_to_config_value(member_value)}\n"
                    )

    @classmethod
    def __convert_member_name_to_config_name(cls, member_name: str) -> str:
        return member_name.replace("_", "-")

    @classmethod
    def __convert_member_value_to_config_value(cls, member_value: AllowedTypesT) -> str:
        if isinstance(member_value, list):
            return " ".join(member_value)

        if isinstance(member_value, bool):
            return "yes" if member_value else "no"

        if isinstance(member_value, Url):
            return member_value.as_string(with_protocol=False)

        if isinstance(member_value, Path):
            return member_value.as_posix()

        return str(member_value)
