export const capitalizeFirst = (text: string) => {
  return text.charAt(0).toUpperCase() + text.slice(1);
};

export const addSpacesAndCapitalizeFirst = (text: string) => {
  return capitalizeFirst(text).replaceAll("_", " ");
};

export const isJson = (item: unknown) => {
  try {
    JSON.parse(item as string);
  } catch (error) {
    return false;
  }

  let value = typeof item !== "string" ? JSON.stringify(item) : item;

  try {
    value = JSON.parse(value);
  } catch (e) {
    return false;
  }

  return (
    typeof value === "object" &&
    value !== null &&
    JSON.stringify(value).includes("{")
  );
};

/**
 * properly formats json including nested json objects
 *
 * @param json json object
 * @returns formatted and parsed json
 */
export const formatJson = (json: { [key: string]: any }) => {
  let formatted = structuredClone(json);
  json &&
    Object.keys(json).forEach((key) => {
      if (typeof json[key] === "object") {
        formatted[key] = formatJson(json[key]);
      } else {
        try {
          formatted[key] = JSON.parse(json[key]);
        } catch (error) {
          formatted[key] = json[key];
        }
      }
    });
  return formatted;
};

export const toDateNumber = (value: number) => {
  if (value < 10) {
    return "0" + value;
  } else {
    return value.toString();
  }
};

/**
 * function to keep path and query string in a URL with decoded reserved charcters
 *
 * @param path path including all needed interpolation params
 * @param query query string as object
 * @returns string containing full path with decoded reserved characters
 */
export const buildDecodedURL = (path: string, query: Object) => {
  let url = `${path}?`;
  Object.entries(query).forEach(([key, value]) => {
    url += `&${key}=${value}`;
  });
  return url;
};

export const formatAccountName = (accountName: string | string[]) => {
  if (Array.isArray(accountName)) {
    return accountName[0].replace("@", "");
  } else {
    return accountName.replace("@", "");
  }
};

export const numberToTimeString = (num: number) => {
  if (num < 10) {
    return `0${num.toString()}`;
  } else {
    return num.toString();
  }
};

export const trimAccountName = (accountName: string) => {
  const trimmedName = accountName.trim();

  if (trimmedName[0] === "@") {
    return trimmedName.slice(1, accountName.length);
  }
  return trimmedName;
};

/**
 * Returns only numbers as string type and trims last word such as `HIVE` or `VESTS`
 *
 * @param value string value that needs to be trimmed
 * @param keyword string keyword of what we want to split from our string
 * @returns trimmed string without keyword
 */

export const splitStringValue = (value: string, keyword: string) => {
  return value.split(keyword)[0];
};

/**
 * Use this to change 123.123 HBD into 123.123 $
 * @param hbd Formatted HBD
 */
export const changeHBDToDollarsDisplay = (hbd: string): string => {
  const numericValue = hbd.split(" ")[0].slice(0, -1);

  return `${numericValue} $`
}

/**
 * 
 * @param hbd grab pure numeric value without any unit or commas
 * @returns 
 */
export const grabNumericValue = (hbd: string): number => {
  const numericValue = Number(hbd.split(" ")[0].slice(0, -1).replaceAll(",", ""));
  return numericValue;
}