from __future__ import annotations

import datetime as dt
from pathlib import Path
import socket
from typing import Final

import pytest
from pytest_postgresql import factories
from sqlalchemy.engine import URL

from benchmark_results_collector.private import common, sync_log_mode
from benchmark_results_collector.private.db_adapter import Db
from benchmark_results_collector.private.sync_log_mode import InfoType
from tests.collect_benchmark_results_tests.constants import PATH_OF_MIXED_SYNC_LOGS

DB_SCHEMA: Final = Path(__file__).parent.parent.parent / 'db/db-schema.sql'

postgresql_external = factories.postgresql_noproc()
postgresql = factories.postgresql('postgresql_external', load=[str(DB_SCHEMA)])


def mock_mapped(
    measurement_timestamp: dt.datetime,
    caller: str,
    method: str,
    params: str,
    value: int,
    unit: str,
    id_: int = None,
) -> common.DbData:
    # pylint: disable=too-many-arguments

    mapped = common.DbData(measurement_timestamp, caller, method, params, value, unit)
    mapped.id_ = id_
    return mapped


def testcase_single_content(measurement_timestamp: dt.datetime, id_: int) -> common.DbData:
    return mock_mapped(
        measurement_timestamp=measurement_timestamp,
        caller='bridge',
        method='get_account_posts',
        params='{"sort": "replies", "account": "gtg", "observer": "gtg"}',
        value=74,
        unit='ms',
        id_=id_,
    )


def testcase_4(id_: int) -> common.DbData:
    return mock_mapped(
        measurement_timestamp=common.convert_millis_to_datetime(1645112818053),
        caller='bridge',
        method='get_community',
        params='{"name": "hive-135485"}',
        value=15,
        unit='ms',
        id_=id_,
    )


def testcase_5(id_: int) -> common.DbData:
    return mock_mapped(
        measurement_timestamp=common.convert_millis_to_datetime(1645112818198),
        caller='bridge',
        method='get_account_posts',
        params='{"sort": "blog", "account": "steemit"}',
        value=26,
        unit='ms',
        id_=id_,
    )


@pytest.fixture(scope='session')
def testcase_timestamps() -> dict[str, dt.datetime]:
    return {
        '1': common.convert_millis_to_datetime(1645112817915),
        '2': common.convert_millis_to_datetime(1645112817927),
        '3': common.convert_millis_to_datetime(1645112817989),
        '4': common.convert_millis_to_datetime(1645112818053),
        '5': common.convert_millis_to_datetime(1645112818198),
    }


@pytest.fixture
def mock_mapped_list(testcase_timestamps) -> list[common.DbData]:
    # pylint: disable=redefined-outer-name

    return [
        testcase_single_content(measurement_timestamp=testcase_timestamps['1'], id_=1),
        testcase_single_content(measurement_timestamp=testcase_timestamps['2'], id_=1),
        testcase_single_content(measurement_timestamp=testcase_timestamps['3'], id_=1),
        testcase_4(id_=1),
        testcase_5(id_=1),
    ]


@pytest.fixture
def mock_mapped_list_distinguished(testcase_timestamps) -> list[common.DbData]:
    # pylint: disable=redefined-outer-name

    return [
        testcase_single_content(measurement_timestamp=testcase_timestamps['1'], id_=1),
        testcase_single_content(measurement_timestamp=testcase_timestamps['2'], id_=2),
        testcase_single_content(measurement_timestamp=testcase_timestamps['3'], id_=3),
        testcase_4(id_=1),
        testcase_5(id_=1),
    ]


@pytest.fixture()
def mock_testcase_row() -> dict[str, str]:
    return {
        'caller': 'bridge',
        'method': 'get_account_posts',
        'parameters': '{"sort": "replies", "account": "gtg", "observer": "gtg"}',
        'hash': '3fb95b06c2116b63740dfabf971380a26d0612934eeebf990ba033fd3aa28e75',
    }


@pytest.mark.asyncio
@pytest.fixture()
async def database(postgresql) -> Db:
    # pylint: disable=redefined-outer-name

    config = {
        'drivername': 'postgresql',
        'username': postgresql.info.user,
        'password': postgresql.info.password,
        'host': postgresql.info.host,
        'port': postgresql.info.port,
        'database': postgresql.info.dbname,
    }
    db_ = await Db.create(URL.create(**config))

    yield db_
    db_.close()
    await db_.wait_closed()


@pytest.fixture()
def sql_select_all() -> str:
    return (
        "SELECT b.description, "
        " b.execution_environment_description,"
        " b.timestamp,"
        " b.server_name,"
        " b.app_version,"
        " b.testsuite_version,"
        " b.runner,"
        " t.caller,"
        " t.method,"
        " t.params,"
        " t.hash,"
        " t.zscore_limit,"
        " bv.measurement_timestamp,"
        " bv.value,"
        " bv.unit"
        " FROM"
        " public.benchmark_values bv"
        " JOIN public.benchmark_description b ON (bv.benchmark_description_id=b.job_id)"
        " JOIN public.testcase t ON (bv.testcase_hash = t.hash)"
    )


@pytest.fixture()
def interesting_sync_log_strings() -> dict[InfoType, list]:
    text = common.get_text_from_log_file(PATH_OF_MIXED_SYNC_LOGS)
    return sync_log_mode.extract_interesting_log_strings(text)


@pytest.fixture()
def sample_measurement() -> dict:
    return {
        'block_number': 5000000,
        'real_ms': 877896,
        'cpu_ms': 560396,
        'current_mem': 7183280,
        'peak_mem': 7183280,
        'index_memory_details_cntr': [],
    }


@pytest.fixture()
def startup_args() -> list[str]:
    # fmt: off
    return ['-j', '22',
            '-db', '',
            '--desc', 'functional test',
            '--exec-env-desc', 'mock db',
            '--server-name', 'localhost',
            '--app-version', '1.0',
            '--testsuite-version', '2.0',
            ]
    # fmt: on


@pytest.fixture()
def benchmark():
    def _benchmark(timestamp: dt.datetime):
        return (
            'functional test',
            'mock db',
            timestamp,
            'localhost',
            '1.0',
            '2.0',
            socket.gethostname(),
        )

    return _benchmark


@pytest.fixture(autouse=True)
def reset_sync_info_class_variable():
    sync_log_mode.SyncInfo.last_block_number = 0
