from __future__ import annotations

import argparse
import asyncio
import datetime as dt
from decimal import Decimal
from enum import Enum
import logging
from pathlib import Path
import smtplib
import sys
from time import perf_counter
from typing import Final, Optional, Sequence
import urllib.parse

import requests
from rocketchat_API.rocketchat import RocketChat

from benchmark_results_collector.private import common
from benchmark_results_collector.private.db_adapter import Db
from benchmark_results_collector.private.logger import logger_setup
from benchmark_results_collector.private.smtp_adapter import SMTP

DEFAULT_DOWNLOAD_PATH: Final[Path] = Path().home()

SMTP_SERVER: Final[str] = 'mail.storm.pl'
ROCKET_CHAT_SERVER: Final[str] = 'https://chat.syncad.com/'
ROCKET_CHAT_CHANNEL_ID: Final[str] = '6SMYqD2hejmzz4WsE'

log = logging.getLogger('benchmark_results_collector')


class Destination(Enum):
    ROCKET_CHAT = 'rocket_chat'
    EMAIL = 'email'


def init_argparse(args: Sequence[str]) -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Check z_score's and send alerts", formatter_class=argparse.RawTextHelpFormatter
    )

    # fmt:off
    add = parser.add_argument
    add('--bearer-token', type=str, help='Grafana api key bearer token.')
    add('--notify', type=str, nargs='*', help='Email addresses to be notified. Multiple allowed.')
    add('-z', '--zscore-limit', type=float, default=None, help='Max acceptable z_score, which not triggers alert. '
                                                               '(absolute value)\n'
                                                               "Default: 'None' (for each testcase, the table value "
                                                               'assigned to it will be compared)')
    add('--interval', type=str, default='30d', help='Time period during which the measurements will be taken into'
                                                    " account for the calculation of the zscore\nDefault: '30d'")
    add('--email', type=str, help='Email address of the account which sends alerts.\n'
                                  'If not provided - alerts will not be sent')
    add('--password', type=str, help='Password of the account which sends alerts.\n'
                                     'If not provided - alerts will not be sent')
    add('--desc-like', type=str, default='%', help='`description` column look, to be included in the measurements.\n'
                                                   "Default: '%%'")
    add('--exec-env-desc-like', type=str, default='%develop%', help='`execution_environment_description` column look,'
                                                                    ' to be included in the measurements.\n'
                                                                    "Default: '%%develop%%'")
    add('--download-path', type=str, help='Folder path where grafana charts will be downloaded')

    req = parser.add_argument_group('required arguments')
    add = req.add_argument
    add('-j', '--job-id', type=int, required=True, help='Job (benchmark) ID.')
    add('--database-url', type=str, required=True, metavar='URL', help='Database URL.')
    # fmt:on

    return parser.parse_args(args)


def grafana_renderer_image_download(url: str, bearer_token: str, save_path: Path, testcase_number: int) -> Path:
    response = requests.get(url, headers={'Authorization': f'Bearer {bearer_token}'})
    save_path /= Path(f'{testcase_number}.png')
    with open(save_path, 'wb') as file:
        file.write(response.content)
    log.info(f'Image saved to: {save_path}')
    return save_path


def get_grafana_renderer_url(testcase_hash: str, args: argparse.Namespace, measurement_timestamp: dt.datetime) -> str:
    args = html_encode_args(args)
    return (
        f'http://benchmark-results.pl.syncad.com:3000/render/d-solo/J7MP5gA7k/testcase-by-hash?orgId=1'
        f'&var-hash={testcase_hash}'
        f"&var-desc_like={args['desc_like']}"
        f"&var-exec_env_desc_like={args['exec_env_desc_like']}"
        f"&var-zscore_interval={args['interval']}"
        f"&var-highlighted_timestamp={common.convert_datetime_to_millis(datetime_=measurement_timestamp)}"
        f"&from=now-{args['interval']}"
        f'&to=now'
        f'&panelId=19'
        f'&width=1920'
        f'&height=1080'
        f'&tz=Europe%2FWarsaw'
    )


def get_grafana_dashboard_url(testcase_hash: str, args: argparse.Namespace, measurement_timestamp: dt.datetime) -> str:
    # escaping bug on the rocketchat side, in progress
    # https://github.com/RocketChat/Rocket.Chat/issues/24417
    # -- START WORKAROUND --
    # args = html_encode_args(args)  # when the problem is solved uncomment this and delete rest
    args = vars(args).copy()
    for key, value in args.items():
        if isinstance(value, str):
            args[key] = value.replace(' ', '+')
    # -- END OF THE WORKAROUND --
    return (
        f'http://benchmark-results.pl.syncad.com:3000/d/J7MP5gA7k/testcase-by-hash?orgId=1'
        f'&var-hash={testcase_hash}'
        f"&var-desc_like={args['desc_like']}"
        f"&var-exec_env_desc_like={args['exec_env_desc_like']}"
        f"&var-zscore_interval={args['interval']}"
        f"&var-highlighted_timestamp={common.convert_datetime_to_millis(datetime_=measurement_timestamp)}"
        f"&from=now-{args['interval']}"
        f'&to=now'
    )


def html_encode_args(args: argparse.Namespace) -> dict:
    args = vars(args).copy()
    for key, value in args.items():
        args[key] = urllib.parse.quote(string=str(value), safe='-=')
    return args


def convert_dict_decimals_to_floats_with_k_places(dictionary: dict, k: int = 2):
    for key in dictionary:
        if isinstance(dictionary[key], Decimal):
            dictionary[key] = float(round(dictionary[key], k))


def get_message(
    testcase: dict,
    measurement: dict,
    benchmark_description: dict,
    zscore_limit: float,
    args: argparse.Namespace,
    destination: Destination,
) -> str:
    # pylint: disable=too-many-arguments

    dashboard_url = get_grafana_dashboard_url(
        testcase_hash=testcase['hash'],
        args=args,
        measurement_timestamp=measurement['measurement_timestamp'],
    )
    if destination is Destination.ROCKET_CHAT:
        link = f":arrow_right: **[GRAFANA DASHBOARD]({dashboard_url})** :arrow_left:"
        intro = '@all\n:warning: '
    else:
        link = f"<a href=\"{dashboard_url}\">[GRAFANA DASHBOARD]</a>"
        intro = ''

    message = (
        f"{intro}"
        f"The testcase with hash `{testcase['hash']}` has z_score [<b>{measurement['z_score']}</b>]"
        f" above the limit [<b>{zscore_limit}</b>]<br><br>"
        f'<b>BENCHMARK DETAILS</b><br>'
        f"<b>job id:</b> {benchmark_description['job_id']}<br>"
        f"<b>timestamp:</b> {benchmark_description['timestamp']}<br>"
        f"<b>description:</b> {benchmark_description['description']}<br>"
        f'<b>execution environment description:</b>'
        f" {benchmark_description['execution_environment_description']}<br>"
        f"<b>server name:</b> {benchmark_description['server_name']}<br>"
        f"<b>app version:</b> {benchmark_description['app_version']}<br>"
        f"<b>testsuite version:</b> {benchmark_description['testsuite_version']}<br><br>"
        f'<b>TESTCASE DETAILS</b><br>'
        f"<b>caller:</b> {testcase['caller']}<br>"
        f"<b>method:</b> {testcase['method']}<br>"
        f"<b>params:</b> {testcase['params']}<br><br>"
        f"<b>MEASUREMENT DETAILS</b><br>"
        f"<b>measurement timestamp:</b> {measurement['measurement_timestamp']}<br>"
        f"<b>last value:</b> {measurement['last_value']}<br>"
        f"<b>avg value:</b> {measurement['avg_value']}<br>"
        f"<b>stddev value:</b> {measurement['stddev_value']}<br><br>"
        f'{link}'
    )

    return (
        message.replace('<b>', '**').replace('</b>', '**').replace('<br>', '\n')
        if destination is destination.ROCKET_CHAT
        else message
    )


def connect_with_rocket_chat(username: str, password: str) -> Optional[RocketChat]:
    if username and password:
        return RocketChat(user=username, password=password, server_url=ROCKET_CHAT_SERVER)
    return None


def connect_with_smtp_server(username: str, password: str) -> Optional[SMTP]:
    try:
        if username and password:
            return SMTP(username=username, password=password, server=SMTP_SERVER)
    except smtplib.SMTPAuthenticationError:
        log.warning('EMAIL AUTHENTICATION ERROR, EMAILS WILL NOT BE SENT')
    return None


async def main():
    start = perf_counter()
    args = init_argparse(sys.argv[1:])

    rocket = connect_with_rocket_chat(username=args.email, password=args.password)
    smtp = connect_with_smtp_server(username=args.email, password=args.password)

    database = await Db.create(args.database_url)
    testcases = await database.query_all(f'SELECT * from get_testcases_by_job_id({args.job_id});')
    benchmark_desc = dict(await database.query_row(f"SELECT * FROM benchmark_description WHERE job_id={args.job_id}"))
    log.info(f'benchmark description is {benchmark_desc}')

    for number, testcase in enumerate(testcases):
        log.info(f'================| TESTCASE no. {number} |================')

        testcase = dict(testcase)
        measurement = dict(
            await database.query_row(
                f"SELECT * FROM get_testcase_zscore"
                f"('{testcase['hash']}',"
                f"'{args.interval}',"
                f"'{args.desc_like}',"
                f"'{args.exec_env_desc_like}')"
            )
        )

        convert_dict_decimals_to_floats_with_k_places(measurement, 3)
        log.debug(f"Z_SCORE={measurement['z_score']}  --- testcase is {testcase}")

        if measurement['z_score'] is None:  # when z_score cannot be calculated (no rows or stddev=0)
            log.warning('measurement Z_SCORE could not be calculated')
            continue

        zscore_limit = args.zscore_limit or float(testcase['zscore_limit'])

        if abs(measurement['z_score']) <= zscore_limit:
            continue

        log.warning(
            f"Z_SCORE [{measurement['z_score']}] over LIMIT [{zscore_limit}] value  --- measurement is {measurement}"
        )

        if not args.bearer_token:
            log.warning('CHART WILL NOT BE DOWNLOADED BECAUSE LAUNCH PARAMETER IS MISSING')
            log.warning(f'--bearer-token={args.bearer_token}')
            continue

        url = get_grafana_renderer_url(
            testcase_hash=testcase['hash'],
            args=args,
            measurement_timestamp=measurement['measurement_timestamp'],
        )
        log.info(f'GRAFANA RENDERER URL IS {url}')

        save_path = grafana_renderer_image_download(
            url=url,
            bearer_token=args.bearer_token,
            save_path=Path(args.download_path) if args.download_path else DEFAULT_DOWNLOAD_PATH,
            testcase_number=number,
        )

        if not rocket:
            log.warning('ALERT WILL NOT BE SENT')
            log.info(f"--email={args.email};{' PASSWORD IS MISSING;' if not args.password else ''}")
            continue

        rocket_message = get_message(
            testcase=testcase,
            measurement=measurement,
            benchmark_description=benchmark_desc,
            zscore_limit=zscore_limit,
            args=args,
            destination=Destination.ROCKET_CHAT,
        )
        rocket.rooms_upload(rid=ROCKET_CHAT_CHANNEL_ID, file=save_path, msg=rocket_message)
        log.info('ROCKETCHAT MESSAGE HAS BEEN SENT')

        if not (args.notify and smtp):
            log.warning('EMAIL MESSAGE WILL NOT BE SENT')
            log.info(
                f"--email={args.email};"
                f"{' PASSWORD IS MISSING;' if not args.password else ''}"
                f" --notify={args.notify}"
            )
            continue

        email_message = get_message(
            testcase=testcase,
            measurement=measurement,
            benchmark_description=benchmark_desc,
            zscore_limit=zscore_limit,
            args=args,
            destination=Destination.EMAIL,
        )
        smtp.send_mail(
            send_from='benchmark-results-collector',
            send_to=args.notify,
            subject=f"Testcase {testcase['hash']} [z_score={measurement['z_score']}]",
            message=email_message,
            files=[save_path],
        )
        log.info(f'EMAIL MESSAGE HAS BEEN SENT TO {args.notify}')

    database.close()
    await database.wait_closed()

    if smtp:
        smtp.close()

    log.info(f'Execution time: {perf_counter() - start:.6f}s')


if __name__ == '__main__':
    logger_setup()
    asyncio.run(main())
