from __future__ import annotations

import json
import logging
from pathlib import Path
import re
from typing import Optional

from benchmark_results_collector.private import common
from benchmark_results_collector.private.db_adapter import Db

log = logging.getLogger('benchmark_results_collector')


def parse_log_line(line: str) -> Optional[dict]:
    match = re.match(r'(\d+) Request: (.*) processed in ([.\d]+)(.*)', line)
    if not match:
        return None

    millis = int(match[1])

    try:
        request = json.loads(match[2])
    except json.JSONDecodeError:
        log.info(f' | [REJECTED FROM PARSING]={match[0]}')
        return None

    if request['method'] == 'call':
        if isinstance(request['params'], list):
            api = request['params'][0]
            method = request['params'][1]
            params = '' if len(request['params']) == 2 else request['params'][2]
        else:
            api = request['params']['api']
            method = request['params']['method']
            params = request['params']['params']
    else:
        api, method = request['method'].split('.')
        params = request['params']

    params_str = json.dumps(params)
    total_time_int = round(float(match[3]) * 10**3) if match[4] == 's' else round(float(match[3]))
    unit = 'ms' if match[4] == 's' else 'unknown'

    return {
        'measurement_timestamp': common.convert_millis_to_datetime(millis),
        'caller': api,
        'method': method,
        'params': params_str,
        'value': total_time_int,
        'unit': unit,
    }


def parse_and_map_log_lines(lines: list[str]) -> list[common.DbData]:
    return [common.DbData(**parsed) for line in lines if (parsed := parse_log_line(line))]


async def main(database: Db, file: Path, benchmark_id: int):
    log_lines = common.get_lines_from_log_file(file)
    mapped_instances = parse_and_map_log_lines(log_lines)

    common.DbData.distinguish_objects_having_same_hash(mapped_instances)

    for mapped in mapped_instances:
        await mapped.insert(database, benchmark_id)
