"""Async DB adapter for hivemind API."""

import logging
from time import perf_counter as perf

from aiopg.sa import create_engine
import sqlalchemy.dialects
from sqlalchemy.engine.url import make_url

log = logging.getLogger('benchmark_results_collector')


class Db:
    """Wrapper for aiopg.sa db driver."""

    @classmethod
    async def create(cls, url):
        """Factory method."""
        instance = Db()
        await instance.init(url)
        return instance

    def __init__(self):
        self.database = None
        self._prep_sql = {}

    async def init(self, url):
        """Initialize the aiopg.sa engine."""
        conf = make_url(url)
        dsn = {}
        if conf.username:
            dsn['user'] = conf.username
        if conf.database:
            dsn['database'] = conf.database
        if conf.password:
            dsn['password'] = conf.password
        if conf.host:
            dsn['host'] = conf.host
        if conf.port:
            dsn['port'] = conf.port
        self.database = await create_engine(**dsn, maxsize=20, minsize=8, **conf.query)

    def close(self):
        """Close pool."""
        self.database.close()

    async def wait_closed(self):
        """Wait for releasing and closing all acquired connections."""
        await self.database.wait_closed()

    async def query_one(self, sql, **kwargs):
        """Perform a `SELECT 1*1`"""
        async with self.database.acquire() as conn:
            cur = await self._query(conn, sql, **kwargs)
            row = await cur.first()
        return row[0] if row else None

    async def query_row(self, sql, **kwargs):
        """Perform a `SELECT 1*m`"""
        async with self.database.acquire() as conn:
            cur = await self._query(conn, sql, **kwargs)
            return await cur.first()

    async def query_all(self, sql, **kwargs):
        """Perform a `SELECT n*m`"""
        async with self.database.acquire() as conn:
            cur = await self._query(conn, sql, **kwargs)
            return await cur.fetchall()

    async def query(self, sql, **kwargs):
        """Perform a write query"""
        async with self.database.acquire() as conn:
            return await self._query(conn, sql, **kwargs)

    @staticmethod
    async def _query(conn, sql, **kwargs):
        """Send a query off to SQLAlchemy."""
        try:
            before = perf()

            # when dialect is set as above - cannot pass datetime object, when dialect is not set - cannot execute fun
            try:
                sql = str(
                    sqlalchemy.text(sql)
                    .bindparams(**kwargs)
                    .compile(dialect=sqlalchemy.dialects.postgresql.dialect(), compile_kwargs={"literal_binds": True})
                )
                result = await conn.execute(sql)
            except NotImplementedError:
                sql = str(sqlalchemy.text(sql).bindparams(**kwargs).compile(compile_kwargs={"literal_binds": True}))
                result = await conn.execute(sql)

            log.info(f'{perf() - before:.6f}s - {sql}')
            return result
        except Exception as exception:
            log.warning(f'[SQL-ERR] {exception.__class__.__name__} in query {sql}')
            raise exception
