from __future__ import annotations

import argparse
import dataclasses
import datetime as dt
from hashlib import sha256
from pathlib import Path
import socket
from typing import Optional

from benchmark_results_collector.private.db_adapter import Db


@dataclasses.dataclass
class DbData:
    # pylint: disable=too-many-instance-attributes

    measurement_timestamp: dt.datetime
    caller: str
    method: str
    params: str
    value: int
    unit: str
    id_: int = dataclasses.field(init=False)
    __hash: str = dataclasses.field(init=False)

    def __post_init__(self):
        self.id_ = 1  # set when function `distinguish_objects_having_same_hash()` is called
        self.__hash = self.calculate_hash(self.caller, self.method, self.params)

    @property
    def hash_(self) -> str:
        return self.__hash

    async def insert(self, database: Db, benchmark_id: int) -> None:
        await self.__insert_into_testcase_table(database)
        await self.__insert_into_benchmark_values_table(database, benchmark_id)

    @staticmethod
    def calculate_hash(*args) -> str:
        return sha256(str(args).encode('utf-8')).hexdigest()

    @staticmethod
    def distinguish_objects_having_same_hash(objects: list) -> None:
        same_by_hash = {}

        for obj in objects:
            if (hash_ := obj.hash_) not in same_by_hash:
                same_by_hash[hash_] = [obj]
            else:
                same_by_hash[hash_].append(obj)

        for lst in same_by_hash.values():
            id_ = 1
            for parsed in lst:
                parsed.id_ = id_
                id_ += 1

    async def __insert_into_testcase_table(self, database: Db) -> None:
        await insert_row(
            database=database,
            table='public.testcase',
            cols_args={
                'hash': self.hash_,
                'caller': self.caller,
                'method': self.method,
                'params': self.params,
            },
            additional=' ON CONFLICT (hash) DO NOTHING;',
        )

    async def __insert_into_benchmark_values_table(self, database: Db, benchmark_id: int) -> None:
        await insert_row(
            database=database,
            table='public.benchmark_values',
            cols_args={
                'benchmark_description_id': benchmark_id,
                'testcase_hash': self.hash_,
                'occurrence_number': self.id_,
                'measurement_timestamp': self.measurement_timestamp,
                'value': self.value,
                'unit': self.unit,
            },
        )


async def insert_benchmark_description(database: Db, args: argparse.Namespace, timestamp: dt.datetime):
    await insert_row(
        database=database,
        table='public.benchmark_description',
        cols_args={
            'job_id': args.job_id,
            'description': args.desc,
            'execution_environment_description': args.exec_env_desc,
            'timestamp': timestamp,
            'server_name': args.server_name,
            'app_version': args.app_version,
            'testsuite_version': args.testsuite_version,
            'runner': socket.gethostname(),
        },
        additional=' ON CONFLICT (job_id) DO NOTHING',
    )


def get_lines_from_log_file(file_path: Path) -> list[str]:
    with open(file_path, 'r', encoding='utf-8') as file:
        return file.readlines()


def get_text_from_log_file(file_path: Path) -> str:
    with open(file_path, 'r', encoding='utf-8') as file:
        return file.read()


def retrieve_cols_and_params(cols_args: dict[str, str]) -> tuple[str, str]:
    """
    Parse dict of cols_args into a two separated strings formats that are needed when
    building a SQL for '_query' method of db_adapter.
    """
    fields = list(cols_args.keys())
    cols = ', '.join(fields)
    params = ', '.join([f':{k}' for k in fields])
    return cols, params


async def insert_row(database: Db, table: str, cols_args: dict, additional: str = '') -> None:
    cols, params = retrieve_cols_and_params(cols_args)
    sql = f'INSERT INTO {table} ({cols}) VALUES ({params}) {additional};'
    await database.query(sql, **cols_args)


def convert_millis_to_datetime(millis: int, tz_: Optional[dt.timezone] = None) -> dt.datetime:
    return dt.datetime.fromtimestamp(millis / 1000, tz=dt.timezone.utc).replace(tzinfo=tz_)


def convert_datetime_to_millis(datetime_: dt.datetime) -> int:
    return round(datetime_.replace(tzinfo=dt.timezone.utc).timestamp() * 10**3)
