from __future__ import annotations

import argparse
import asyncio
import datetime as dt
import json
import logging
import sys
from time import perf_counter
from typing import Final, Sequence, Union
import urllib.parse

import requests

from benchmark_results_collector.private import common
from benchmark_results_collector.private.db_adapter import Db
from benchmark_results_collector.private.logger import logger_setup

log = logging.getLogger('benchmark_results_collector')

GRAFANA_URL: Final[str] = 'http://benchmark-results.pl.syncad.com:3000/'
ANNOTATIONS_ENDPOINT: Final[str] = urllib.parse.urljoin(base=GRAFANA_URL, url='/api/annotations/')


def init_argparse(args: Sequence[str]) -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description='Add annotations with benchmark description details' ' to the given grafana dashboard',
        formatter_class=argparse.RawTextHelpFormatter,
    )

    # fmt: off
    req = parser.add_argument_group('required arguments')
    add = req.add_argument
    add('-j', '--job-id', type=int, required=True, help='Job (benchmark) ID.')
    add('--database-url', type=str, required=True, metavar='URL', help='Database URL.')
    add('--bearer-token', type=str, required=True, help='Grafana api key bearer token.')

    add = parser.add_argument
    add('--tags', type=str, nargs='+', help='Tags included in annotation.')
    add('--dashboard-id', type=int, help='If not specified then a global annotation is created and can be queried'
                                         ' in any dashboard that adds the Grafana annotations data source')
    add('--panel-id', type=int, help='If not specified then a global annotation is created and can be queried'
                                     ' in any dashboard that adds the Grafana annotations data source')
    # fmt: on

    return parser.parse_args(args)


def get_all_annotations(bearer_token: str, params: str = None) -> Union[dict, list]:
    response = requests.get(
        url=urllib.parse.urljoin(base=ANNOTATIONS_ENDPOINT, url=params),
        headers={'Authorization': f'Bearer {bearer_token}'},
    ).json()

    for annotation in response:
        parse_annotation_timestamps_to_datetimes(annotation)

    return response


def parse_annotation_timestamps_to_datetimes(annotation: dict) -> dict:
    annotation['created'] = common.convert_millis_to_datetime(annotation['created'], tz_=dt.timezone.utc)
    annotation['updated'] = common.convert_millis_to_datetime(annotation['updated'], tz_=dt.timezone.utc)
    annotation['time'] = common.convert_millis_to_datetime(annotation['time'], tz_=dt.timezone.utc)
    annotation['timeEnd'] = common.convert_millis_to_datetime(annotation['timeEnd'], tz_=dt.timezone.utc)
    return annotation


def post_annotation(bearer_token: str, data: dict) -> dict:
    headers = {
        'Authorization': f'Bearer {bearer_token}',
        'Content-Type': 'application/json',
    }

    return requests.post(url=ANNOTATIONS_ENDPOINT, headers=headers, data=json.dumps(data)).json()


def create_annotation_body(timestamp: int, args: argparse.Namespace, benchmark_description: dict):
    dashboard_id = {'dashboardId': args.dashboard_id} if args.dashboard_id else {}
    panel_id = {'panelId': args.panel_id} if args.panel_id else {}

    return {
        **dashboard_id,
        **panel_id,
        'time': timestamp,
        'text': create_annotation_text(benchmark_description),
        'tags': args.tags,
    }


def create_annotation_text(benchmark_description):
    return (
        f"Release [{benchmark_description['app_version']}]<br>"
        f"description: [{benchmark_description['description']}]<br>"
        f"exec env description: [{benchmark_description['execution_environment_description']}]<br>"
        f"app version: [{benchmark_description['app_version']}]<br>"
        f"testsuite version: [{benchmark_description['testsuite_version']}]"
    )


async def main():
    start = perf_counter()
    args = init_argparse(sys.argv[1:])
    timestamp = common.convert_datetime_to_millis(dt.datetime.now(tz=dt.timezone.utc))

    database = await Db.create(args.database_url)

    benchmark_description = await database.query_row(f"SELECT * FROM benchmark_description WHERE id={args.job_id}")
    log.info(f'Benchmark description: {benchmark_description}')

    annotation = create_annotation_body(timestamp=timestamp, args=args, benchmark_description=benchmark_description)

    response = post_annotation(bearer_token=args.bearer_token, data=annotation)
    log.info(f"Annotation with ID {response['id']} added")

    added = get_all_annotations(bearer_token=args.bearer_token, params=f'?from={timestamp}&to={timestamp}')
    log.info(f'\n{json.dumps(added, indent=4, sort_keys=True, default=str)}')

    database.close()
    await database.wait_closed()

    log.info(f'Execution time: {perf_counter() - start:.6f}s')


if __name__ == '__main__':
    logger_setup(level=logging.DEBUG)
    asyncio.run(main())
