import argparse
import asyncio
import datetime as dt
import logging
from pathlib import Path
import sys
from time import perf_counter
from typing import Sequence

from benchmark_results_collector.private import common, replay_benchmark_mode, server_log_mode, sync_log_mode
from benchmark_results_collector.private.db_adapter import Db
from benchmark_results_collector.private.logger import logger_setup

log = logging.getLogger('benchmark_results_collector')


def init_argparse(args: Sequence[str]) -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description='Parse a benchmark log file.', formatter_class=argparse.RawTextHelpFormatter
    )

    # fmt: off
    add = parser.add_argument
    add('--desc', type=str, default='', help='Benchmark description.')
    add('--exec-env-desc', type=str, default='', help='Execution environment description.')
    add('--server-name', type=str, default='', help='Server name when benchmark has been performed')
    add('--app-version', type=str, default='')
    add('--testsuite-version', type=str, default='')

    req = parser.add_argument_group('required arguments')
    add = req.add_argument
    add('-j', '--job-id', type=int, required=True, help='Job (benchmark) ID.')
    add('-f', '--file', type=str, required=True, metavar='FILE_PATH', help='Source .log / .json file path.')
    add('-db', '--database-url', type=str, required=True, metavar='URL', help='Database URL.')
    add('-m', '--mode', type=int, required=True, choices=[1, 2, 3], help='1 - SERVER_LOG,\n'
                                                                         '2 - SYNC_LOG,\n'
                                                                         ' 3 - REPLAY_BENCHMARK')
    # fmt: on
    return parser.parse_args(args)


async def main():
    start = perf_counter()
    timestamp = dt.datetime.now(dt.timezone.utc)
    timestamp = timestamp.replace(microsecond=round(timestamp.microsecond / 1000) * 1000)  # 1ms precision

    args = init_argparse(sys.argv[1:])
    log.info(f'[START ARGS]={vars(args)}')
    log.info('------------------------------------------------------')

    database = await Db.create(args.database_url)

    await common.insert_benchmark_description(database, args, timestamp)

    if args.mode == 1:
        log.info('[MODE]=SERVER_LOG')
        await server_log_mode.main(database=database, file=Path(args.file), benchmark_id=args.job_id)
    elif args.mode == 2:
        log.info('[MODE]=SYNC_LOG')
        # no multiple testcases with same hash for one benchmark_description, so we don't need to distinguish them
        await sync_log_mode.main(
            database=database,
            file=Path(args.file),
            benchmark_id=args.job_id,
            measurement_timestamp=timestamp,
        )
    elif args.mode == 3:
        log.info('[MODE]=REPLAY_BENCHMARK')
        # no multiple testcases with same hash for one benchmark_description, so we don't need to distinguish them
        await replay_benchmark_mode.main(
            database=database,
            file=Path(args.file),
            benchmark_id=args.job_id,
            measurement_timestamp=timestamp,
        )

    database.close()
    await database.wait_closed()
    log.info(f'Execution time: {perf_counter() - start:.6f}s')


if __name__ == '__main__':
    logger_setup()
    asyncio.run(main())
